<?php // $Id: textmagic.class.php,v 1.3 2011/06/19 01:28:24 janne Exp $
/**
 * This page holds SMSTextmagic sub class.
 * @package SMS
 * @subpackage SMSTextmagic
 */

/**
* Textmagic subclass
*
* @package SMS
* @subpackage SMSTextmagic
* @author Janne Mikkonen
* @version $Revision: 1.3 $
* @copyright 2011 Janne Mikkonen
* @license http://opensource.org/licenses/gpl-license.php GNU Public License
*/
class SMSTextmagic extends SMS {

    /**
     * TextMagic's HTTP API commands
     * We're generally supporting send command.
     *
     * @var string
     */
    private $_cmd = 'send';

    /**
     * Recipients number
     *
     * Can be a string or an array
     * @var mixed
     */
    private $_varto = 'phone';
    private $_varfrom = 'from';
    private $_vartext = 'text';
    private $_recipientdelim = ',';
    private $_unicode = 0;

    /**
     * Detect unicode chars.
     *
     * Try to detect unicode characters
     * automatically. 
     * @var bool
     */
    public  $probeunicode = true;

    /**
     * Request URI to API app.
     *
     * @var string
     */
    private $_scriptpath = '/app/api';
    /**
     * Username fieldname NOT the username it self...
     * @var string
     */
    private $_username = 'username';

    /**
     * Password fieldname NOT the password it self.
     * @var string 
     */
    private $_password = 'password';
   
    /**
     * Constructor.
     * @param object $config
     */
    public function __construct(&$config) {
         parent::initialize($config);
    }

    /**
     * Set request URI to HTTP call.
     *
     * @uses PARAM_LOCALURL Moodle's variable cleaning system.
     * @param string $path URI to receiving app.
     * @return void
     */
    public function set_script_path($path) {
        $path = clean_param($path, PARAM_LOCALURL);
        $this->_scriptpath = $path;
    }

    /**
    * Send SMS message through TextMagic's HTTPS gateway web interface.
    *
    * @param mixed $to Receivers mobilephone number or array of numbers.
    * @param string $text SMS message to send.
    * @param string $from Senders mobilephone number.
    * @return mixed Returns false on failure and HTTP response on success.
    */

    public function send_message($to, $text, $from=null) {

        // Check if recipient address is an array meaning
        // there are multiple recipients for this message.
        if ( is_array($to) && $this->_recipientdelim !== false ) {
            $tmp = '';
            for ( $i = 0; $i < sizeof($to); $i++ ) {
                if ( $i > 0 ) {
                    $tmp .= $this->_recipientdelim;
                }
                $tmp .= $this->clean_number($to[$i]);
            }
            $to = $tmp;
        } else {
            $to = $this->clean_number($to);
        }

        if ( $this->probeunicode ) {
            if ( strlen($text) !== strlen(utf8_decode($text)) ) {
                $this->_unicode = 1;
            }
        }

        $data = new stdClass;
        $data->to   = $to;
        $data->text = $text;
        $data->from = $from;
        $data->account = $this->clientid;

        // Kannel have username and password variables and it doesn't use
        // basic authentication method so, we'll need to add these values
        // into query string.
        $data->username = !empty($this->gwuser) ? $this->gwuser : '';
        $data->password = !empty($this->gwpass) ? $this->gwpass : '';

        if (! empty($auth) ) {
            $data->auth = $auth;
        }

        $command = $this->_build_send_cmd ($data);

        if ($result = $this->__exec($command)) {
            try {
                return $this->_decode($result);
            } catch (Exception $e) {
                printf("%s\r\n", $e->getMessage());
            }
        } else {
            return false;
        }

    }

    /**
    * Not implemented.
    */
    function send_mms_message ($to, $data) {
        trigger_error("This method is not implemented!", E_USER_ERROR);
    }

    /**
     * Build request string fro HTTP GET method.
     *
     * @param object $data
     * @return string
     */
    private function _build_send_cmd ($data) {

        $to       = '';
        $from     = '';
        $text     = '';
        $username = '';
        $password = '';

        if ( !empty($data->username) ) {
            $username = $data->username;
        }

        if ( !empty($data->password) ) {
            $password = $data->password;
        }

        if ( !empty($data->to) ) {
            $to = urlencode($data->to);
        }

        if ( !empty($data->from) ) {
            $from = rawurlencode($data->from);
        }

        if ( !empty($data->text) ) {
            $text = rawurlencode($data->text);
        }

        $account = !empty($data->account) ? $data->account : $this->clientid;

        $postdata = sprintf("%s=%s&%s=%s&%s=%s&%s=%s&%s=%s&%s=%d",
                        'cmd', 
                        $this->_cmd,
                        $this->_username,
                        $username,
                        $this->_password,
                        $password,
                        $this->_varto,
                        $to,
                        $this->_vartext,
                        $text,
                        'unicode',
                        $this->_unicode);
        

        if ( !empty($from) ) {
            $postdata .= sprintf("&%s=%s",
                                $this->_varfrom,
                                $from);
        }

        $command = $this->_build_post_request($postdata);

        return $command;

    }

    /**
     * Generate HTTP GET request
     * 
     * Generate HTTP GET request string. Data it self
     * must be in proper form (key=value pairs and separated
     * with & if more than one key=value pairs are given).
     * @param string $data
     * @return string
     */
    private function _build_get_request($getdata) {

        $command = null;

        switch ($this->socket) {
            case 'socket':
            case 'fsockopen':
                // Generate HTTP 1.1 Request string.
                $command  = sprintf("GET %s?%s HTTP/1.1\r\nHost: %s:%d\r\n",
                                    $this->_scriptpath, 
                                    $getdata, 
                                    $this->gwhost, 
                                    $this->gwport);
                $command .= sprintf("User-Agent: %s\r\n", $this->useragent);
                $command .= sprintf("Connection: %s\r\n", 'close');
                $command .= "\r\n"; // Terminate request with empty line.
                // Add data part.
                $command .= $postdata;
            break;
            case 'curl':
            break;
        }
        return $command;
    }

    /**
     * Generate POST request
     * 
     * Generate POST reqeust with content-type of
     * application/x-www-form-urlencoded
     * @param string $postdata
     * @return string
     */
    private function _build_post_request($postdata) {

        $command = null;
        $content_length = strlen($postdata);
        switch ($this->socket) {
            case 'socket':
            case 'fsockopen':
                // Generate HTTP 1.1 Request string.
                $command  = sprintf("POST %s HTTP/1.1\r\nHost: %s:%d\r\n",
                                    $this->_scriptpath, $this->gwhost, $this->gwport);
                $command .= sprintf("Content-Type: %s\r\n",  'application/x-www-form-urlencoded');
                $command .= sprintf("Content-Length: %d\r\n", $content_length);
                $command .= sprintf("User-Agent: %s\r\n", $this->useragent);
                $command .= sprintf("Connection: %s\r\n", 'close');
                $command .= "\r\n"; // Terminate request with empty line.
                // Add data part.
                $command .= $postdata;
            break;
            case 'curl': break;
        }
    
        return $command;

    }

    public function setUnicode(int $unicode) {
        $this->_unicode = $unicode;
    }

    private function _getResponseLength($response) {
        if ( preg_match("/Content-Length: ([0-9]+)/", $response, $match) ) {
            return (int) $match[1];
        }
        return 0;
    }

    /**
     * Get the body data from HTTP response
     *
     * @param string $data The whole response as a string
     * @param integer $length Content-Length value of response.
     * @return string Body part as a string.
     */
    public function getResponseBody($data) {
        if ( !$length = $this->_getResponseLength($data) ) {
            $error = "Could not get proper length value from response body!";
            throw new Exception($error);
        }
        
        $total = strlen($data);
        if ( $total < $length ) {
            throw new Exception("Requested length is larger than total ({$total}) length!");
        }
        $start = $total - $length;
        return substr($data, $start, $length);

    }
   
    /**
     * Get message id or id's from JSON response.
     *
     * @param string $response Response AS JSON string
     * @return mixed
     */
    public function getMessageId($response) {
        if ( function_exists('json_decode') ) {
            $response = json_decode($response);
            if ( !empty($response->message_id) ) {
                return $response->message_id;
            }
        } else {
            throw new Exception("JSON extension is not available!");
            return false;
        }
        return 0;
    }

    /**
     * Get account balance
     * @return float
     */
    public function getBalance() {

        $getreq  = sprintf("%s=%s",  $this->_username, $this->gwuser);
        $getreq .= sprintf("&%s=%s", $this->_password, $this->gwpass);
        $getreq .= sprintf("&cmd=%s", 'account');

        $command = $this->_build_get_request($getreq);

        if ($result = $this->__exec($command)) {

            try {
                $response = $this->_decode($result);

                if ( !empty($response->balance) ) {
                    return (float) $response->balance;
                }

            } catch (Exception $e) {
                printf("%s\r\n", $e->getMessage());
            }

        } else {
            return 0;
        }

    }

    /**
     * Get message status/statuses
     *
     * Returns an array of message status or statuses
     * @param mixed $ids Message ids.
     * @return array
     */
    public function getMessageStatus($ids) {

        if ( is_array($ids) ) {
            $ids = implode(",", $ids);
        }

        if ( !is_string($ids) ) {
            throw new Exception('Wrong param type! Should be string or an array!');
        }

        $getreq  = sprintf("%s=%s", $this->_username, $this->gwuser);
        $getreq .= sprintf("&%s=%s", $this->_password, $this->gwpass);
        $getreq .= sprintf("&cmd=%s", 'message_status');
        $getreq .= sprintf("&ids=%s", $ids);

        $command = $this->_build_get_request($getreq);

        if ( $result = $this->__exec($command) ) {
            try {
                return $this->_decode($result, true);
            } catch (Exception $e) {
                printf("%s\r\n", $e->getMessage());
            }
        } 
        return false;
        
    }

    /**
     * Get received SMS messages.
     *
     * Get received SMS messages from TextMagic's
     * server. Returns an object of message arrays. 
     *
     * @param int $last_retrieved_id Last id of retrieved message id.
     * @return mixed
     */
    public function getMessages($last_retrieved_id=0) {

        $getreq  = sprintf("%s=%s", $this->_username, $this->gwuser);
        $getreq .= sprintf("&%s=%s", $this->_password, $this->gwpass);
        $getreq .= sprintf("&cmd=%s", 'receive');
        $getreq .= sprintf("&last_retrieved_id=%d", $last_retrieved_id);

        $command = $this->_build_get_request($getreq);

        if ( $result = $this->__exec($command) ) {
            try {
                return $this->_decode($result);
            } catch (Exception $e) {
                printf("%s\r\n", $e->getMessage());
            }
        }
        return false;

    }

    /**
     * Delete reply's from TextMagic's server
     *
     * Returns a object of arrays containing id's of
     * deleted messages.
     *
     * @param mixed $ids a single id or an array of id's
     * @return mixed
     */
    public function deleteReply($ids) {
        
        $dopost = false;

        if ( is_array($ids) ) {
            if ( count($ids) > 10 ) {
                $dopost = true;
            }
            $ids = implode(",", $ids);
        }
        
        $req  = sprintf("%s=%s", $this->_username, $this->gwuser);
        $req .= sprintf("&%s=%s", $this->_password, $this->gwpass);
        $req .= sprintf("&cmd=%s", 'delete_reply');
        $req .= sprintf("&ids=%s", $ids);

        $command = ($dopost) ? $this->_build_post_request($req):
                               $this->_build_get_request($req);


        if ( $result = $this->__exec($command) ) {
            try {
                return $this->_decode($result);
            } catch (Exception $e) {
                printf("%s\r\n", $e->getMessage());
            }
        }
        return false;

    }

    /**
     * Check number and get price and country info
     * of the number/numbers
     *
     * Returns an array.
     * @param mixed $number a single number or an array of numbers.
     * @return mixed
     */
    public function checkNumber ($number) {

        $dopost = false;

        if ( is_array($number) ) {
            if ( count($number) > 10 ) {
                $dopost = true;
            }
            $number = implode(",", $number);
        }
        
        $req  = sprintf("%s=%s", $this->_username, $this->gwuser);
        $req .= sprintf("&%s=%s", $this->_password, $this->gwpass);
        $req .= sprintf("&cmd=%s", 'check_number');
        $req .= sprintf("&phone=%s", $number);

        $command = ($dopost) ? $this->_build_post_request($req):
                               $this->_build_get_request($req);

        if ( $result = $this->__exec($command) ) {
            try {
                return $this->_decode($result, true);
            } catch (Exception $e) {
                printf("%s\r\n", $e->getMessage());
            }
        }
        return false;

    }

    /**
     * Wrapper for JSON decode.
     *
     * Wrapper for JSON decode. Try to detect
     * possible error messages as well.
     *
     * @param string $response HTTP response body part
     * @return mixed
     */
    private function _decode($response,$assoc=false,$depth=512) {
        if ( !function_exists('json_decode') ) {
            die('JSON Extension is not present! Enable or install JSON extension.');
        }

        // Check if headers are still present.
        if ( preg_match("/^HTTP\/1\.(0|1)/", $response) ) {
            $response = $this->getResponseBody($response);
        }

        $result = false;
        try {
            $result = json_decode($response, $assoc, $depth);
        } catch (ErrorException $e) {
            printf("%s\r\n", $e->getMessage());
            die;
        }
        
        if ( !$assoc and !empty($result->error_code) ) {
            throw new Exception("{$result->error_code} {$result->error_message}");
        }
        if ( $assoc and !empty($result['error_code']) ) {
            throw new Exception("{$result['error_code']} {$result['error_message']}");
        }

        return $result;

    }

    /**
     * Get message from $_POST
     *
     * Get incoming message from superglobal $_POST
     * array. This can be used when TextMagic's API
     * callback url for incoming messages is set to some
     * script in your server.
     * @return object Return incoming message as an object.
     */
    public function parseIncoming() {

        if ( !empty($_POST['message_id']) ) {
            $icm = new stdClass;
            $icm->message_id = preg_replace("/[^0-9]/", "", $_POST['message_id']);
            $icm->timestamp = preg_replace("/[^0-9]/", "", $_POST['timestamp']);
            $icm->from = preg_replace("/[^0-9]/", "", $_POST['from']);
            $icm->text = trim(strip_tags($_POST['text']));
            return $icm;
        }
        return false;

    }

    // End of class.
}
?>
